<?php
/**
 * User Level
 *
 * @package     AutomatorWP\Integrations\WishList_Member\Actions\User_Level
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_WishList_Member_User_Level extends AutomatorWP_Integration_Action {

    public $integration = 'wishlist_member';
    public $action = 'wishlist_member_user_level';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Add or remove level from user', 'automatorwp-pro' ),
            'select_option'     => __( 'Add or remove <strong>level</strong> from user', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Level. */
            'edit_label'        => sprintf( __( '%1$s %2$s to user', 'automatorwp-pro' ), '{operation}', '{level}' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Level. */
            'log_label'         => sprintf( __( '%1$s %2$s to user', 'automatorwp-pro' ), '{operation}', '{level}' ),
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'add'       => __( 'Add', 'automatorwp-pro' ),
                                'remove'    => __( 'Remove', 'automatorwp-pro' ),
                            ),
                            'default' => 'add'
                        ),
                    )
                ),
                'level' => array(
                    'from' => 'level',
                    'default' => __( 'Select a level', 'automatorwp-pro' ),
                    'fields' => array(
                        'level' => array(
                            'name' => __( 'Level:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'classes' => 'automatorwp-selector',
                            'option_custom'         => true,
                            'option_custom_value'   => 'custom',
                            'option_custom_label'   => __( 'Use a custom value', 'automatorwp' ),
                            'option_custom_desc'    => '',
                            'options_cb' => array( $this, 'options_cb_levels' ),
                            'default' => '',
                        ),
                        'level_custom' => automatorwp_utilities_custom_field( array(
                            'option_custom_desc' => __( 'Level ID', 'automatorwp-pro' ),
                        ) )
                    )
                )
            ),
        ) );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Dynamic edit and log labels
        add_filter( 'automatorwp_parse_automation_item_edit_label', array( $this, 'dynamic_label' ), 10, 5 );
        add_filter( 'automatorwp_parse_automation_item_log_label', array( $this, 'dynamic_label' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Custom edit/log label
     *
     * @since 1.0.0
     *
     * @param string    $label      The edit label
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The item type (trigger|action)
     * @param string    $context    The context this function is executed
     * @param array     $type_args  The type parameters
     *
     * @return string
     */
    public function dynamic_label( $label, $object, $item_type, $context, $type_args ) {

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $label;
        }

        // Get the operation value
        ct_setup_table( "automatorwp_{$item_type}s" );
        $operation = ct_get_object_meta( $object->id, 'operation', true );
        ct_reset_setup_table();

        // Update the edit label
        if( $operation === 'remove' ) {
            return sprintf( __( '%1$s %2$s from user', 'automatorwp-pro' ), '{operation}', '{level}' );
        }

        return $label;

    }

    /**
     * Options callback for levels options
     *
     * @since 1.0.0
     *
     * @param stdClass $field
     *
     * @return array
     */
    public function options_cb_levels( $field ) {

        $options = array(
            'any' => __( 'all levels', 'automatorwp-pro' ),
            'custom' => __( 'Use a custom value', 'automatorwp' ),
        );

        if( function_exists( 'wlmapi_get_levels' ) ) {

            // Get all registered levels
            $levels = wlmapi_get_levels();

            // Check that levels are correctly setup
            if ( is_array( $levels )
                && isset( $levels['levels'] )
                && isset( $levels['levels']['level'] )
                && ! empty( $levels['levels']['level'] ) ) {

                // Loop levels to add them as options
                foreach ( $levels['levels']['level'] as $level ) {
                    $options[$level['id']] = $level['name'];
                }

            }

        }

        return $options;

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        global $WishListMemberInstance;

        // Shorthand
        $operation = $action_options['operation'];
        $level = $action_options['level'];

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'add';
        }

        $levels = array();
        $all_levels = $WishListMemberInstance->get_option( 'wpm_levels' );
        $user_levels = $WishListMemberInstance->get_membership_levels( $user_id );

        // Check specific level
        if( $level !== 'any' ) {

            // Bail if level doesn't exists
            if( ! isset( $all_levels[$level] ) ) {
                return;
            }

            $levels = array( $level );
        }

        switch( $operation ) {
            case 'add':
                // If adding to all levels, get all stored levels
                if( $level === 'any' ) {
                    $levels = wp_list_pluck( $all_levels, 'id' );
                }

                // Add levels to the user
                foreach( $levels as $level_id ) {

                    // If user already on this level, skip
                    if( in_array( $level_id, $user_levels ) ) {
                        continue;
                    }

                    $user_levels[] = $level_id;

                    // Add level to user
                    $WishListMemberInstance->SetMembershipLevels( $user_id, $user_levels );
                    // Update level transaction ID to meet from where comes
                    $WishListMemberInstance->SetMembershipLevelTxnID( $user_id, $level_id, "AutomatorWP-action-{$action->id}-automation-{$automation->id}-" . current_time( 'timestamp' ) );

                }
                breaK;
            case 'remove':
                // If removing from all levels, get user levels
                if( $level === 'any' ) {
                    $levels = $user_levels;
                }

                // Remove levels to the user
                foreach( $levels as $level_id ) {

                    // Check if user has this level
                    $index = array_search( $level_id, $user_levels );

                    if ( $index !== false ) {
                        // Remove level
                        unset( $user_levels[$index] );
                    }

                }

                // Update user levels and sync them
                $WishListMemberInstance->set_membership_levels( $user_id, $user_levels );
                $WishListMemberInstance->schedule_sync_membership(true);
                break;
        }

    }

}

new AutomatorWP_WishList_Member_User_Level();